MacroSequence = {}

function MacroSequence:Initialize(sequences)
  if not sequences then
    DEFAULT_CHAT_FRAME:AddMessage("MacroSequencesSequences does not exist. Please make sure there are no errors in Interface\\AddOns\\MacroSequence\\Sequences.lua.")
    return
  end

  for name, data in pairs(sequences) do
    self:CreateSequence(name, data)
  end
end

function MacroSequence:CreateSequence(name, data)
  if #data == 0 then
    DEFAULT_CHAT_FRAME:AddMessage(format("MacroSequence %s contains no macros. See Interface\\AddOns\\MacroSequence\\Sequences.lua.", name))
    return
  end
  
  -- Create the state header for the sequence
  local header = CreateFrame("Frame", nil, nil, "SecureStateHeaderTemplate")
  
  -- Create the actual button for clicking
  local button = CreateFrame(
    "Button",
    name,
    header,
    "SecureActionButtonTemplate"
  )
  header:SetAttribute("addchild", button)
  button:SetAttribute("type", "macro")

  -- Set up a newstate attribute to cycle through each possible state
  button:SetAttribute("newstate", "0-"..(#data - 1))
  
  -- Add entries to the sequence
  self:CreateSequenceEntries(button, data)
  
  -- Set up reset conditions for the sequence
  self:ApplyResetConditions(header, button, data)
end

function MacroSequence:CreateSequenceEntries(button, data)
  local statebutton = ""
  for state, macro in ipairs(data) do
    state = state - 1
    -- Add a statebutton entry for the current macro in the form "0:B0;"
    statebutton = statebutton..format("%d:B%1$d;", state)
    -- Create a macrotext attribute using the new statebutton
    button:SetAttribute("*macrotext-B"..state, macro)
  end
  button:SetAttribute("statebutton", statebutton)
end

function MacroSequence:ApplyResetConditions(header, button, data)
  if #data == 1 then
    -- Reset is meaningless if there's only one entry
    return
  end
  
  -- Make the sequence reset after death and combat (if necessary)
  button:SetScript("OnEvent", function(button, event)
    if event == "PLAYER_DEAD" then
      if InCombatLockdown() then
        button.resetDeath = true
      else
        header:SetAttribute("state", "0")
      end
    elseif event == "PLAYER_REGEN_ENABLED" then
      if button.resetCombat or button.resetDeath then
        header:SetAttribute("state", "0")
        button.resetDeath = nil
      end
    end
  end)
  button:RegisterEvent("PLAYER_DEAD")
  button:RegisterEvent("PLAYER_REGEN_ENABLED")

  if not data.reset then
    return
  end
  
  button.resetCombat = data.reset.combat
  self:ApplyTimerReset(button, data)
  self:ApplyModifierResets(button, data)
end

function MacroSequence:ApplyTimerReset(button, data)
  local seconds = data.reset.seconds
  if seconds and seconds > 0 then
    -- Switch to state 0 after the specified delay unless the state changes by
    -- other means first
    button:SetAttribute("delaystate", "0")
    button:SetAttribute("delaytime", seconds)
  end
end

-- This table describes which prefixes to use for each reset modifier's newstate
-- attribute. For the sake of simplicity, ApplyModifierResets will rewrite
-- certain attributes if multiple modifier resets are used on a single sequence.
local modifierPrefixes = {
  alt = {
    "alt",
    "alt-ctrl",
    "alt-shift",
    "alt-ctrl-shift"
  },
  ctrl = {
    "ctrl",
    "alt-ctrl",
    "ctrl-shift",
    "alt-ctrl-shift"
  },
  shift = {
    "shift",
    "alt-shift",
    "ctrl-shift",
    "alt-ctrl-shift"
  }
}
function MacroSequence:ApplyModifierResets(button, data)
  for modifier, prefixes in pairs(modifierPrefixes) do
    if data.reset[modifier] then
      for _, prefix in ipairs(prefixes) do
        -- Make the modified click run the first macro in the sequence
        button:SetAttribute(prefix.."-statebutton*", "B0")
        -- And then move on to the next
        button:SetAttribute(prefix.."-newstate*", "1")
      end
    end
  end
end

MacroSequence:Initialize(MacroSequenceSequences)
