TargetText = {}

--[[----------------------------------------------------------------------------
"Constants"
----------------------------------------------------------------------------]]--
-- Table of events for registering & unregistering when the frame is
-- shown/hidden
local events = {
  "UNIT_NAME_UPDATE",
  "UNIT_LEVEL",
  "UNIT_CLASSIFICATION_CHANGED",
  "UNIT_FACTION",
  "UNIT_PVP_UPDATE",
  "UNIT_HEALTH",
  "UNIT_MAXHEALTH",
  "UNIT_DISPLAYPOWER",
  "UNIT_MANA",
  "UNIT_MAXMANA",
  "UNIT_RAGE",
  "UNIT_MAXRAGE",
  "UNIT_ENERGY",
  "UNIT_MAXENERGY",
  "UNIT_FOCUS",
  "UNIT_MAXFOCUS",
  "UNIT_HAPPINESS",
  "UNIT_MAXHAPPINES",
}

-- Colors we'll use that don't exist in Fonts.xml
local colors = {
  white = { r = 1, g = 1, b = 1},
  blue = { r = 0, g = 0, b = 1}
}

-- Descriptions of the various reaction levels
local reactionNames = {
  "Extremely Hostile",
  "Very Hostile",
  "Hostile",
  "Neutral",
  "Friendly",
  "Very Friendly",
  "Extremely Friendly"
}

-- "Constants" for manually set reaction colors
local HOSTILE = 2
local NEUTRAL = 4
local FRIENDLY = 6

--[[----------------------------------------------------------------------------
Functions
----------------------------------------------------------------------------]]--
function TargetText:OnLoad()
  -- Register fundamental events
  TargetTextFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
  TargetTextFrame:RegisterEvent("PLAYER_TARGET_CHANGED")
  -- Make clickable
  TargetTextFrame:RegisterForClicks("AnyUp")
  -- Show/hide when gaining/losing a target
  RegisterUnitWatch(TargetTextFrame)
  
  -- Set up behaviors
  CogsUtils:GetPrint("test")
  self:SetManyAttributes(TargetTextFrame,
    "type", "spell",
    
    -- Unmodified left click
    "helpbutton1",  "heal",
    "*spell-heal",  "Greater Heal",
    "harmbutton1",  "blast",
    "*spell-blast", "Mind Blast",
    
    -- Unmodified right click
    "type2", "focus",
    
    -- Shift left click
    "shift-helpbutton1", "renew",
    "*spell-renew",      "Renew",
    "shift-harmbutton1", "pain",
    "*spell-pain",       "Shadow Word: Pain",
    
    -- Shift right click
    "shift-unit2",  "focus",
    "shift-spell2", "Shackle Undead"
  )
end

function TargetText:OnEvent(event, unit)
  -- Fundamental event to show/hide the frame
  if event == "PLAYER_ENTERING_WORLD" or event == "PLAYER_TARGET_CHANGED" then
    self:UpdateAll()
    
  -- Events registered when the frame is shown
  elseif unit == "target" then
    if event == "UNIT_NAME_UPDATE" then
      self:UpdateName()
    elseif event == "UNIT_LEVEL" or event == "UNIT_CLASSIFICATION_CHANGED" then
      self:UpdateLevel()
    elseif event == "UNIT_FACTION" then
      self:UpdateFaction()
      self:UpdateReaction()
      self:UpdateLevel() -- Level will not have changed, but target may now be
                         -- hostile so we want to change the color of the level
    elseif event == "UNIT_PVP_UPDATE" then
      self:UpdatePvP()
    elseif event == "UNIT_HEALTH" or event == "UNIT_MAXHEALTH" then
      self:UpdateHealth()
    elseif event == "UNIT_DISPLAYPOWER" then
      self:UpdatePowerType()
    elseif event == "UNIT_MANA" or
        event == "UNIT_MAXMANA" or
        event == "UNIT_RAGE" or
        event == "UNIT_MAXRAGE" or 
        event == "UNIT_ENERGY" or
        event == "UNIT_MAXENERGY" or
        event == "UNIT_FOCUS" or
        event == "UNIT_MAXFOCUS" or 
        event == "UNIT_HAPPINESS" or
        event == "UNIT_MAXHAPPINESS" then
      self:UpdateMana()
    end
  end
end

-- Called in OnShow
function TargetText:RegisterUpdates()
  for _, event in ipairs(events) do
    TargetTextFrame:RegisterEvent(event)
  end
end

-- Called in OnHide
function TargetText:UnregisterUpdates()
  for _, event in ipairs(events) do
    TargetTextFrame:UnregisterEvent(event)
  end
end

function TargetText:UpdateAll()
  if UnitExists("target") then
    self:UpdateName()
    self:UpdateLevel()
    self:UpdateClass()
    self:UpdateReaction()
    self:UpdateFaction()
    self:UpdatePvP()
    self:UpdateHealth()
    self:UpdatePowerType()
    self:UpdateMana()
  end
end

function TargetText:UpdateName()
  local name = UnitName("target")
  TargetTextFrameName:SetText(name)
end

function TargetText:UpdateLevel()
  local level = UnitLevel("target")
  
  -- If the target is attackable, color the level based on difficulty
  local color
  if UnitCanAttack("player", "target") then
    color = GetDifficultyColor(level)
  else
    color = colors.white
  end
  
  -- Obfuscated level means big scary creature
  if level == -1 then
    level = "??"
  end
  
  -- Add a '+' to elite mobs
  local classification = UnitClassification("target")
  if classification == "worldboss" or
    classification == "rareelite" or
    classification == "elite" then
    
    level = level.."+"
  end
  
  TargetTextFrameLevel:SetText(level)
  TargetTextFrameLevel:SetTextColor(color.r, color.g, color.b)
end

function TargetText:UpdateClass()
  local class, englishClass = UnitClass("target")
  local color = RAID_CLASS_COLORS[englishClass]
  
  TargetTextFrameClass:SetText(class)
  TargetTextFrameClass:SetTextColor(color.r, color.g, color.b)
end

function TargetText:UpdatePvP()
  if UnitIsPVP("target") then
    TargetTextFramePvP:Show()
  else
    TargetTextFramePvP:Hide()
  end
end

function TargetText:UpdateReaction()
  -- Default description & color
  local description, color = "No Reaction", colors.blue
  
  if UnitIsDead("target") then
    color = GRAY_FONT_COLOR
    if UnitIsCorpse("target") then
      description = "Corpse"
    elseif UnitIsGhost("target") then
      description = "Ghost"
    else
      -- Just in case a target can be dead but neither a ghost nor a corpse
      description = "Dead"
    end
  elseif UnitPlayerControlled("target") then
    if not UnitIsConnected("target") then
      -- Gray if target is disconnected
      description = "Disconnected"
      color = GRAY_FONT_COLOR
    elseif UnitCanAttack("target", "player") and UnitCanAttack("player", "target") then
      -- Red for enemy PvP players who can attack you
      description = reactionNames[HOSTILE]
      color = UnitReactionColor[HOSTILE]
    elseif UnitCanAttack("player", "target") then
      -- Yellow for enemy PvP players who cannot attack you
      description = "Attackable"
      color = UnitReactionColor[NEUTRAL]
    elseif UnitIsPVP("target") and not UnitIsPVPSanctuary("target") and not UnitIsPVPSanctuary("player") then
      -- Green for friendly PvP players
      description = reactionNames[FRIENDLY]
      color = UnitReactionColor[FRIENDLY]
    end
  elseif UnitIsTapped("target") and not UnitIsTappedByPlayer("target") then
    -- Gray for tapped mobs
    description = "Tapped"
    color = GRAY_FONT_COLOR
  else
    -- All other cases, calculate based on UnitReaction()
    local reaction = UnitReaction("target", "player")
    if reaction then
      description = reactionNames[reaction]
      color = UnitReactionColor[reaction]
    end
  end
  
  TargetTextFrameReaction:SetText(description)
  TargetTextFrameReaction:SetTextColor(color.r, color.g, color.b)
end

function TargetText:UpdateFaction()
  local faction = select(2, UnitFactionGroup("target"))
  local color = NORMAL_FONT_COLOR
  
  if not faction then
    faction = "No Faction"
  elseif faction == select(2, UnitFactionGroup("player")) then
    color = GREEN_FONT_COLOR
  else
    color = RED_FONT_COLOR
  end

  TargetTextFrameFaction:SetText(faction)
  TargetTextFrameFaction:SetTextColor(color.r, color.g, color.b)
end

function TargetText:UpdateHealth()
  local health, maxHealth = UnitHealth("target"), UnitHealthMax("target")
  local percent = health/maxHealth
  local healthText
  
  if maxHealth == 0 then
    healthText = "N/A"
  elseif maxHealth == 100 then
    healthText = health.."%"
  else
    healthText = string.format("%d/%d (%.0f%%)", health, maxHealth, percent * 100)
  end
  
  TargetTextFrameHealth:SetText(healthText)
  TargetTextFrameHealth:SetTextColor(self:GetHealthColor(percent))
end

-- 100% == Green, 50% == Yellow, 0% == Red
function TargetText:GetHealthColor(percent)
  local red, green
  
  if percent >= 0.5 then
    red = (1 - percent) * 2
    green = 1
  else
    red = 1
    green = percent * 2
  end
  
  return red, green, 0
end

function TargetText:UpdatePowerType()
  local info = ManaBarColor[UnitPowerType("target")]
  TargetTextFrameManaLabel:SetText(info.prefix)
  TargetTextFrameMana:SetTextColor(info.r, info.g, info.b)
end

function TargetText:UpdateMana()
  local mana, maxMana = UnitMana("target"), UnitManaMax("target")
  local percent = mana/maxMana
  local manaText
  
  if maxMana == 0 then
    manaText = "N/A"
  else
    manaText = string.format("%d/%d (%.0f%%)", mana, maxMana, percent * 100)
  end
  
  TargetTextFrameMana:SetText(manaText)
end

function TargetText:SetManyAttributes(frame, ...)
  for i = 1, select("#", ...), 2 do
    frame:SetAttribute(select(i, ...), select(i + 1, ...))
  end
end
